DROP PROCEDURE [hips].[PatientInHospitalGet]
GO
SET ANSI_NULLS ON
GO
SET QUOTED_IDENTIFIER ON
GO
-- =============================================
-- Author:        Simon Biber
-- Create date:   20 Nov 2013
-- Description:   Gets a list of patients in hospital who have an active IHI and match other given criteria.
-- =============================================
CREATE PROCEDURE [hips].[PatientInHospitalGet] 
(
	@HospitalCodeSystem VARCHAR(20),
	@HospitalId INT = NULL,
	@WithPcehr BIT = NULL,
	@DaysDischarged INT
)
AS
BEGIN

DECLARE @DischargeDateCutoff DATETIME
SET @DischargeDateCutoff = DATEADD(d, -@DaysDischarged, GETDATE())

-- Temporary table to store ID of the most recent matching episode for each
-- patient who matches the conditions in the parameters to this stored procedure
CREATE TABLE #MatchingEpisodes (
	EpisodeId INT NOT NULL
)

-- In case there are multiple episodes with the same admission date, pick the
-- most recently created episode.
INSERT	#MatchingEpisodes
SELECT	MAX(e.EpisodeId)
FROM hips.Episode e
 INNER JOIN (
	-- Determine the admission date of the most recent episode for each patient who
	-- has a current episode in the required hospital and meets the WithPcehr condition.
	SELECT	e.PatientId,
			MAX(e.AdmissionDate) AS MaxAdmit
	FROM	hips.HospitalPatient hp
	 INNER JOIN hips.Episode e
		ON hp.PatientId = e.PatientId
	 LEFT JOIN hips.HealthProviderOrganisationPatient hpop
		ON hp.PatientMasterId = hpop.PatientMasterId
	WHERE (@HospitalId IS NULL OR hp.HospitalId = @HospitalId)
	  AND (@WithPcehr IS NULL 
			OR   @WithPcehr = '0' AND (    (hpop.PcehrAdvertised = 0 AND hpop.PcehrDisclosed = 0) 
										OR (hpop.HealthProviderOrganisationId IS NULL AND hpop.PatientMasterId IS NULL) 
									  )
			OR   @WithPcehr = '1' AND (hpop.PcehrAdvertised = 1 OR hpop.PcehrDisclosed = 1)
	  )
	AND (	e.EpisodeLifecycleId = 11  -- EpisodeLifecycleId 11 is Admitted, 13 is Discharged
		OR	e.EpisodeLifecycleId = 13 AND e.DischargeDate > @DischargeDateCutoff
		)
	GROUP BY e.PatientId
 ) a 
	ON e.PatientId = a.PatientId
	AND e.AdmissionDate = a.MaxAdmit
WHERE	(	e.EpisodeLifecycleId = 11
		OR	e.EpisodeLifecycleId = 13 AND e.DischargeDate > @DischargeDateCutoff
		)
GROUP BY e.PatientId


SELECT pm.PatientMasterId, 
		pm.CurrentSexId,
		pm.DateOfBirth,
		pm.MedicareNumber,
		pm.MedicareIrn,
		pm.DvaNumber,
		pmn.FamilyName,
		pmn.GivenNames,
		pmnt.Description AS Title,
		pmns.Description AS Suffix,
		pmi.Ihi,
		pmi.IhiStatusId,
		pmi.IhiRecordStatusId,
		pmi.DateLastValidated AS IhiLastValidated,
		pmi.RegisteredSexId,
		pmi.RegisteredFamilyName,
		pmi.RegisteredGivenName,
		hp.HospitalId,
		hp.Mrn,
		hc.Code as HospitalCode,
		h.Name as HospitalName,
		e.Ward,
		e.Room,
		e.Bed,
		e.AdmissionDate,
		e.DischargeDate,
		rp.FamilyName AS ResponsibleProviderFamilyName,
		rp.GivenNames AS ResponsibleProviderGivenNames,
		rpt.Description AS ResponsibleProviderTitle,
		rps.Description AS ResponsibleProviderSuffix,
		hpop.PcehrAdvertised,
		hpop.PcehrDisclosed
FROM #MatchingEpisodes me
 INNER JOIN hips.Episode e
	ON me.EpisodeId = e.EpisodeId
 INNER JOIN hips.HospitalPatient hp
	ON e.PatientId = hp.PatientId
 INNER JOIN hips.PatientMaster pm
    ON hp.PatientMasterId = pm.PatientMasterId
 INNER JOIN hips.PatientMasterName pmn
	ON pm.PatientMasterId = pmn.PatientMasterId
	AND pmn.NameTypeId = 2 -- Current Name
 INNER JOIN hips.PatientMasterIhi pmi
	ON pm.PatientMasterId = pmi.PatientMasterId
	AND pmi.IhiStatusId = 1 -- Active IHI
	AND pmi.IhiRecordStatusId = 1 -- Verified IHI
	AND LEN(pmi.Ihi) = 16 -- Correct Length IHI
 INNER JOIN hips.HospitalCode hc
	ON hp.HospitalId = hc.HospitalId
	AND hc.CodeSystemId = (SELECT CodeSystemId FROM hips.CodeSystem WHERE Code = @HospitalCodeSystem)
 INNER JOIN hips.Hospital h
    ON hp.HospitalId = h.HospitalId
 INNER JOIN hips.HealthProviderIndividual rp
	ON e.ResponsibleProviderId = rp.HealthProviderIndividualId
 INNER JOIN hips.Title pmnt
    ON pmn.TitleId = pmnt.TitleId
 INNER JOIN hips.Suffix pmns
    ON pmn.SuffixId = pmns.SuffixId
 INNER JOIN hips.Title rpt
    ON rp.TitleId = rpt.TitleId
 INNER JOIN hips.Suffix rps
    ON rp.SuffixId = rps.SuffixId
 LEFT JOIN hips.HealthProviderOrganisationPatient hpop
	ON pm.PatientMasterId = hpop.PatientMasterId

END

GO

